Error subclass: #BusinessRuleException
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Collaboration-Examples'!


Model subclass: #Document
	instanceVariableNames: 'title publicationDate securityLevel nominations '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Collaboration-Examples'!

!Document methodsFor: 'initialize-release' stamp: 'mlm 3/17/2001 12:23'!
initialize
	title _ String new.
	publicationDate _ nil.
	securityLevel _ SecurityLevel new.
	nominations _ OrderedCollection new! !


!Document methodsFor: 'accessing' stamp: 'mlm 3/9/2001 10:29'!
publicationDate
	publicationDate ifNil: [BusinessRuleException signal: 'Document is unpublished.'].
	^ publicationDate! !

!Document methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:22'!
securityLevel
	^ securityLevel! !

!Document methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:22'!
title
	^ title! !

!Document methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:08'!
title: aTitleString
	| aTrimmedTitleString |
	(aTitleString isNil or: [(aTrimmedTitleString _ aTitleString withBlanksTrimmed) isEmpty])
		ifTrue: [BusinessRuleException signal: 'Document cannot have nil or empty title.'].
	self testSetTitle: aTrimmedTitleString.
	self doSetTitle: aTrimmedTitleString! !


!Document methodsFor: 'accessing-rules' stamp: 'mlm 3/17/2001 19:45'!
testSetPublicationDate
	self isPublished ifTrue: [BusinessRuleException signal: 'Document already published.'].
	self approvedNomination
		ifNil: [BusinessRuleException signal: 'Document not approved for publication.']! !

!Document methodsFor: 'accessing-rules' stamp: 'mlm 3/17/2001 19:45'!
testSetTitle: aTitleString
	aTitleString size > 255
		ifTrue: [BusinessRuleException signal: 'Document title cannot exceed 255 characters.']! !


!Document methodsFor: 'collaboration-accessing' stamp: 'mlm 3/17/2001 12:33'!
approvedNomination
	^ self nominations detect: [:aNomination | aNomination isStatusApproved]
		ifNone: [BusinessRuleException signal: 'Document has no approved nomination.']! !

!Document methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:23'!
nominations
	^ nominations! !


!Document methodsFor: 'collaboration-rules' stamp: 'mlm 3/17/2001 19:01'!
testAddNomination: aNomination
	self isPublished ifTrue: [BusinessRuleException signal: 'Document already published.'].
	self hasUnresolvedNominations
		ifTrue: [BusinessRuleException signal: 'Document has unresolved nomination.']! !

!Document methodsFor: 'collaboration-rules' stamp: 'mlm 4/11/2001 10:34'!
testAddNominationConflict: aNomination with: aTeamMember
	(self securityLevel > aTeamMember securityLevel) ifTrue: [BusinessRuleException signal:
		'Security violation. Team Member has improper security.']! !


!Document methodsFor: 'comparing' stamp: 'mlm 9/23/2000 11:45'!
= anObject
	self species = anObject species ifFalse: [^ false].
	self title = anObject title ifFalse: [^ false].
	^ true! !


!Document methodsFor: 'domain services' stamp: 'mlm 9/27/2000 13:50'!
nominate: aTeamMember
	Nomination newWith: aTeamMember and: self! !

!Document methodsFor: 'domain services' stamp: 'mlm 3/17/2001 13:18'!
publish
	self testSetPublicationDate.
	self doSetPublicationDate: Date today! !


!Document methodsFor: 'printing' stamp: 'mlm 3/2/2001 09:44'!
printOn: aStream 
	aStream nextPutAll: 'Document:'.
	aStream cr; nextPutAll: self title.
	self securityLevel printOn: aStream cr! !


!Document methodsFor: 'testing' stamp: 'mlm 9/27/2000 13:51'!
hasNomination: aNomination
	^ self nominations includes: aNomination! !

!Document methodsFor: 'testing' stamp: 'mlm 3/17/2001 12:55'!
hasUnresolvedNominations
	^ self nominations anySatisfy: [:aNomination | aNomination isNotResolved]! !

!Document methodsFor: 'testing' stamp: 'mlm 3/17/2001 12:32'!
isApproved
	^ self nominations anySatisfy: [:aNomination | aNomination isStatusApproved]! !

!Document methodsFor: 'testing' stamp: 'mlm 9/27/2000 14:48'!
isPublished
	^ publicationDate notNil! !

!Document methodsFor: 'testing' stamp: 'mlm 9/27/2000 14:47'!
isUnpublished
	^ publicationDate isNil! !


!Document methodsFor: 'private' stamp: 'mlm 12/3/1999 20:34'!
doAddNomination: aNomination
	self nominations add: aNomination.
	self changed: #nominations! !

!Document methodsFor: 'private' stamp: 'mlm 3/17/2001 13:17'!
doSetPublicationDate: aPublicationDate
	publicationDate _ aPublicationDate.
	self changed: #publicationDate! !

!Document methodsFor: 'private' stamp: 'mlm 3/17/2001 12:57'!
doSetTitle: aTitleString
	title _ aTitleString.
	self changed: #title! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Document class
	instanceVariableNames: ''!

!Document class methodsFor: 'instance creation' stamp: 'mlm 9/15/2000 16:07'!
new
	BusinessRuleException signal: 'Cannot create document without a title.'! !

!Document class methodsFor: 'instance creation' stamp: 'mlm 9/23/2000 12:47'!
newWith: aTitleString
	| aDocument |
	aDocument _ super new initialize.
	aDocument title: aTitleString.
	^ aDocument! !


!Document class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:41'!
testNomination1
	"Document testNomination1"
	| document teamMember |
	document _ Document testNormal.
	teamMember _ TeamMember testAdmin.
	[document nominate: teamMember]
		on: BusinessRuleException
		do: [:ex | Transcript cr; show: ex messageText].
	^ document! !

!Document class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:41'!
testNomination2
	"Document testNomination2"
	| document teamMember |
	document _ Document testNormal.
	teamMember _ TeamMember testChair.
	[document nominate: teamMember]
		on: BusinessRuleException
		do: [:ex | Transcript cr; show: ex messageText].
	^ document! !

!Document class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:41'!
testNomination3
	"Document testNomination3"
	| document teamMember |
	document _ Document testNormal.
	teamMember _ TeamMember testNoNominate.
	[document nominate: teamMember]
		on: BusinessRuleException
		do: [:ex | Transcript cr; show: ex messageText].
	^ document! !

!Document class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:41'!
testNomination4
	"Document testNomination4"
	| document teamMember |
	document _ Document testSecret.
	teamMember _ TeamMember testAdmin.
	[document nominate: teamMember]
		on: BusinessRuleException
		do: [:ex | Transcript cr; show: ex messageText].
	^ document! !

!Document class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:42'!
testNormal
	"Document testNormal"
	| aDocument |
	aDocument _ self newWith: 'Normal Document'.
	^ aDocument! !

!Document class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:42'!
testSecret
	"Document testSecret"
	| aDocument |
	aDocument _ self newWith: 'Food and Beverage Industry Surveillance Tips'.
	aDocument securityLevel setLevelSecret.
	^ aDocument! !


Model subclass: #Nomination
	instanceVariableNames: 'status nominationDate comments document teamMember '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Collaboration-Examples'!

!Nomination methodsFor: 'initialize-release' stamp: 'mlm 9/27/2000 13:33'!
initialize
	status _ self statusPending.
	nominationDate _ Date today.
	comments _ String new.
	document _ nil.
	teamMember _ nil! !


!Nomination methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:24'!
comments
	^ comments! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 19:47'!
comments: aCommentString
	aCommentString
		ifNil: [BusinessRuleException signal: 'Tried to add nil comment for nomination.'].
	comments _ aCommentString withBlanksTrimmed.
	self changed: #comments! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:25'!
isStatusApproved
	^ self doGetStatus = self statusApproved! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:25'!
isStatusInReview
	^ self doGetStatus = self statusInReview! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:25'!
isStatusPending
	^ self doGetStatus = self statusPending! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:25'!
isStatusRejected
	^ self doGetStatus = self statusRejected! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:24'!
nominationDate
	^ nominationDate! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:24'!
setStatusApproved
	self testSetStatusApproved.
	self doSetStatus: self statusApproved! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:24'!
setStatusInReview
	self testSetStatusInReview.
	self doSetStatus: self statusInReview! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:24'!
setStatusPending
	self testSetStatusPending.
	self doSetStatus: self statusPending! !

!Nomination methodsFor: 'accessing' stamp: 'mlm 3/17/2001 13:24'!
setStatusRejected
	self testSetStatusRejected.
	self doSetStatus: self statusRejected! !


!Nomination methodsFor: 'accessing-rules' stamp: 'mlm 3/9/2001 10:28'!
testSetStatusApproved
	(self isStatusInReview or: [self isStatusApproved]) ifFalse: [BusinessRuleException signal:
		'Nomination cannot be approved. Not under review.']! !

!Nomination methodsFor: 'accessing-rules' stamp: 'mlm 3/9/2001 10:26'!
testSetStatusInReview
	self isStatusNotResolved ifFalse: [BusinessRuleException signal:
		'Nomination already resolved. Cannot make in review.']! !

!Nomination methodsFor: 'accessing-rules' stamp: 'mlm 3/9/2001 10:25'!
testSetStatusPending
	self isStatusNotResolved ifFalse: [BusinessRuleException signal:
		'Nomination already resolved. Cannot make pending.']! !

!Nomination methodsFor: 'accessing-rules' stamp: 'mlm 3/9/2001 10:25'!
testSetStatusRejected
	(self isStatusInReview or: [self isStatusRejected])
		ifFalse: [BusinessRuleException signal: 'Nomination cannot be rejected. Not under review.']! !


!Nomination methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:26'!
addDocument: aDocument
	aDocument ifNil: [BusinessRuleException signal: 'Tried to add nil document.'].
	self testAddDocument: aDocument.
	aDocument testAddNomination: self.
	self doAddDocument: aDocument.
	aDocument doAddNomination: self! !

!Nomination methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:26'!
addTeamMember: aTeamMember
	aTeamMember ifNil: [BusinessRuleException signal: 'Tried to add nil team member.'].
	self testAddTeamMember: aTeamMember.
	aTeamMember testAddNomination: self.
	self doAddTeamMember: aTeamMember.
	aTeamMember doAddNomination: self! !

!Nomination methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:24'!
document
	^ document! !

!Nomination methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:24'!
teamMember
	^ teamMember! !


!Nomination methodsFor: 'collaboration-rules' stamp: 'mlm 4/11/2001 10:37'!
testAddDocument: aDocument
	self document ifNotNil: [BusinessRuleException signal: 'Document already exists.'].
	self teamMember ifNotNil: [aDocument testAddNominationConflict: self with: self teamMember]! !

!Nomination methodsFor: 'collaboration-rules' stamp: 'mlm 4/11/2001 10:35'!
testAddTeamMember: aTeamMember
	self teamMember ifNotNil: [BusinessRuleException signal: 'Team member already exists.'].
	self document ifNotNil: [self document testAddNominationConflict: self with: aTeamMember]! !


!Nomination methodsFor: 'comparing' stamp: 'mlm 3/17/2001 13:25'!
= anObject 
	self species = anObject species ifFalse: [^ false].
	self doGetStatus = anObject doGetStatus ifFalse: [^ false].
	self nominationDate = anObject nominationDate ifFalse: [^ false].
	self document ifNil: [anObject document ifNotNil: [^ false]].
	self document ifNotNil: [self document = anObject document ifFalse: [^ false]].
	self teamMember ifNil: [anObject teamMember ifNotNil: [^ false]].
	self teamMember ifNotNil: [self teamMember = anObject teamMember ifFalse: [^ false]].
	^ true! !


!Nomination methodsFor: 'constants' stamp: 'mlm 9/25/2000 16:34'!
statusApproved
	^ Association key: #approved value: 'Approved'! !

!Nomination methodsFor: 'constants' stamp: 'mlm 9/25/2000 16:34'!
statusInReview
	^ Association key: #inReview value: 'In Review'! !

!Nomination methodsFor: 'constants' stamp: 'mlm 9/25/2000 16:34'!
statusPending
	^ Association key: #pending value: 'Pending'! !

!Nomination methodsFor: 'constants' stamp: 'mlm 9/25/2000 16:34'!
statusRejected
	^ Association key: #rejected value: 'Rejected'! !


!Nomination methodsFor: 'printing' stamp: 'mlm 3/17/2001 13:25'!
printOn: aStream 
	aStream nextPutAll: 'Nomination:'.
	self nominationDate printOn: aStream cr.
	aStream cr; nextPutAll: self doGetStatus value.
	self teamMember printOn: aStream cr.
	self document printOn: aStream cr! !


!Nomination methodsFor: 'testing' stamp: 'mlm 9/23/2000 11:25'!
isAfter: aDate
	^ aDate < self nominationDate! !

!Nomination methodsFor: 'testing' stamp: 'mlm 9/23/2000 11:25'!
isBefore: aDate
	^ self nominationDate < aDate! !

!Nomination methodsFor: 'testing' stamp: 'mlm 9/25/2000 16:47'!
isStatusNotResolved
	^ self isStatusPending or: [self isStatusInReview]
! !


!Nomination methodsFor: 'private' stamp: 'mlm 1/28/2000 13:42'!
doAddDocument: aDocument
	document _ aDocument.
	self changed: #document! !

!Nomination methodsFor: 'private' stamp: 'mlm 2/27/2001 14:52'!
doAddTeamMember: aTeamMember
	teamMember _ aTeamMember.
	self changed: #teamMember! !

!Nomination methodsFor: 'private' stamp: 'mlm 3/17/2001 13:23'!
doGetStatus
	^ status! !

!Nomination methodsFor: 'private' stamp: 'mlm 3/17/2001 13:23'!
doSetStatus: aStatus
	status _ aStatus.
	self changed: #status! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Nomination class
	instanceVariableNames: ''!

!Nomination class methodsFor: 'instance creation' stamp: 'mlm 9/15/2000 16:35'!
new
	BusinessRuleException
			signal: 'Cannot create nomination without a team member and a document.'! !

!Nomination class methodsFor: 'instance creation' stamp: 'mlm 3/17/2001 12:45'!
newWith: aTeamMember and: aDocument
	| aNomination |
	aNomination _ super new initialize.
	aNomination addTeamMember: aTeamMember.
	[aNomination addDocument: aDocument]
		on: BusinessRuleException
		do: [:ex | aTeamMember doRemoveNomination: aNomination.
			ex signal].
	^ aNomination! !


!Nomination class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:46'!
testNomination
	"Nomination testNomination"
	| aNomination |
	aNomination _ self newWith: TeamMember testChair and: Document testNormal.
	^ aNomination! !


Model subclass: #Person
	instanceVariableNames: 'name title email teamMembers '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Collaboration-Examples'!

!Person methodsFor: 'initialize-release' stamp: 'mlm 9/6/2000 15:12'!
initialize
	name _ String new.
	title _ String new.
	email _ String new.
	teamMembers _ OrderedCollection new! !


!Person methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:30'!
email
	^ email! !

!Person methodsFor: 'accessing' stamp: 'mlm 3/17/2001 10:10'!
email: anEmailAddressString
	| aTrimmedEmailAddressString |
	[MailAddressParser addressesIn:
		(aTrimmedEmailAddressString _ anEmailAddressString withBlanksTrimmed)]
			on: Error do: [:ex | BusinessRuleException signal: 'Bad email address.'].
	email _ aTrimmedEmailAddressString.
	self changed: #email! !

!Person methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:30'!
name
	^ name! !

!Person methodsFor: 'accessing' stamp: 'mlm 3/17/2001 10:00'!
name: aNameString
	| aTrimmedNameString |
	(aNameString isNil or: [(aTrimmedNameString _ aNameString withBlanksTrimmed) isEmpty])
		ifTrue: [BusinessRuleException signal: 'Person name cannot be nil or empty.'].
	name _ aTrimmedNameString.
	self changed: #name! !

!Person methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:30'!
title
	^ title! !

!Person methodsFor: 'accessing' stamp: 'mlm 3/17/2001 10:00'!
title: aTitleString
	aTitleString ifNil: [BusinessRuleException signal: 'Person cannot have nil title.'].
	title _ aTitleString withBlanksTrimmed.
	self changed: #title! !


!Person methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:11'!
addTeamMember: aTeamMember
	aTeamMember ifNil: [BusinessRuleException signal: 'Tried to add nil team member.'].
	aTeamMember addPerson: self! !

!Person methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:11'!
removeTeamMember: aTeamMember
	aTeamMember ifNil: [BusinessRuleException signal: 'Tried to remove nil team member.'].
	aTeamMember removePerson: self! !

!Person methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:11'!
teamMemberFor: aTeam
	^ self teamMembers detect: [:aTeamMember | aTeamMember team = aTeam] ifNone: []! !

!Person methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:30'!
teamMembers
	^ teamMembers! !


!Person methodsFor: 'comparing' stamp: 'mlm 3/17/2001 10:05'!
= anObject 
	self species = anObject species ifFalse: [^ false].
	self name = anObject name ifFalse: [^ false].
	self email = anObject email ifFalse: [^ false].
	self title = anObject title ifFalse: [^ false].
	^ true! !


!Person methodsFor: 'printing' stamp: 'mlm 3/17/2001 10:06'!
printOn: aStream
	aStream nextPutAll: 'Person:'.
	aStream cr; nextPutAll: self name.
	aStream cr; nextPutAll: self title.
	aStream cr; nextPutAll: self email! !


!Person methodsFor: 'testing' stamp: 'mlm 3/9/2001 10:12'!
hasValidEmail
	self email isEmptyOrNil ifTrue: [^ false].
	^ true! !


!Person methodsFor: 'private' stamp: 'mlm 9/6/2000 15:14'!
doAddTeamMember: aTeamMember
	self teamMembers add: aTeamMember.
	self changed: #teamMembers! !

!Person methodsFor: 'private' stamp: 'mlm 9/6/2000 15:18'!
doRemoveTeamMember: aTeamMember
	self teamMembers remove: aTeamMember ifAbsent: [].
	self changed: #teamMembers! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Person class
	instanceVariableNames: ''!

!Person class methodsFor: 'instance creation' stamp: 'mlm 8/21/2000 13:20'!
new
	BusinessRuleException signal: 'Cannot create person without a name.'! !

!Person class methodsFor: 'instance creation' stamp: 'mlm 9/23/2000 12:48'!
newWith: aNameString
	| aPerson |
	aPerson _ super new initialize.
	aPerson name: aNameString.
	^ aPerson! !


!Person class methodsFor: 'examples' stamp: 'mlm 3/17/2001 10:06'!
testPerson
	"Person testPerson"
	| aPerson |
	aPerson _ self newWith: 'Alfred E. Neumann'.
	aPerson email: 'al@neumann.com'.
	aPerson title: 'President'.
	^ aPerson! !


Magnitude subclass: #SecurityLevel
	instanceVariableNames: 'level '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Collaboration-Examples'!

!SecurityLevel methodsFor: 'initialize-release' stamp: 'mlm 3/17/2001 12:09'!
initialize
	self setLevelLow! !


!SecurityLevel methodsFor: 'comparing' stamp: 'mlm 3/17/2001 12:07'!
< aSecurityLevel
	^ self doGetLevel < aSecurityLevel doGetLevel! !

!SecurityLevel methodsFor: 'comparing' stamp: 'mlm 3/17/2001 12:07'!
= anObject 
	self species = anObject species ifFalse: [^ false].
	self doGetLevel = anObject doGetLevel ifFalse: [^ false].
	^ true! !


!SecurityLevel methodsFor: 'constants' stamp: 'mlm 9/27/2000 13:57'!
high
	^ Association key: 3 value: 'High Security'! !

!SecurityLevel methodsFor: 'constants' stamp: 'mlm 9/27/2000 13:57'!
low
	^ Association key: 0 value: 'Low Security'! !

!SecurityLevel methodsFor: 'constants' stamp: 'mlm 9/27/2000 13:57'!
medium
	^ Association key: 2 value: 'Medium Security'! !

!SecurityLevel methodsFor: 'constants' stamp: 'mlm 9/27/2000 14:34'!
secret
	^ Association key: 4 value: 'Secret Security'! !


!SecurityLevel methodsFor: 'domain services' stamp: 'mlm 3/17/2001 11:02'!
setLevelHigh
	self doSetLevel: self high! !

!SecurityLevel methodsFor: 'domain services' stamp: 'mlm 3/17/2001 11:02'!
setLevelLow
	self doSetLevel: self low! !

!SecurityLevel methodsFor: 'domain services' stamp: 'mlm 3/17/2001 11:58'!
setLevelMedium
	self doSetLevel: self medium! !

!SecurityLevel methodsFor: 'domain services' stamp: 'mlm 3/17/2001 11:58'!
setLevelSecret
	self doSetLevel: self secret! !


!SecurityLevel methodsFor: 'printing' stamp: 'mlm 3/17/2001 12:07'!
printOn: aStream
	aStream nextPutAll: self doGetLevel value! !


!SecurityLevel methodsFor: 'testing' stamp: 'mlm 3/17/2001 12:05'!
isLevelHigh
	^ self doGetLevel = self high! !

!SecurityLevel methodsFor: 'testing' stamp: 'mlm 3/17/2001 12:05'!
isLevelLow
	^ self doGetLevel = self low! !

!SecurityLevel methodsFor: 'testing' stamp: 'mlm 3/17/2001 12:05'!
isLevelMedium
	^ self doGetLevel = self medium! !

!SecurityLevel methodsFor: 'testing' stamp: 'mlm 3/17/2001 12:05'!
isLevelSecret
	^ self doGetLevel = self secret! !


!SecurityLevel methodsFor: 'private' stamp: 'mlm 3/17/2001 11:58'!
doGetLevel
	^ level! !

!SecurityLevel methodsFor: 'private' stamp: 'mlm 3/17/2001 11:59'!
doSetLevel: aLevel
	level _ aLevel! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SecurityLevel class
	instanceVariableNames: ''!

!SecurityLevel class methodsFor: 'instance creation' stamp: 'mlm 9/23/2000 12:48'!
new
	^ super new initialize! !


Model subclass: #Team
	instanceVariableNames: 'description teamMembers format '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Collaboration-Examples'!

!Team methodsFor: 'initialize-release' stamp: 'mlm 3/17/2001 19:05'!
initialize
	description _ String new.
	teamMembers _ OrderedCollection new.
	format _ self formatMultipleChair! !


!Team methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:34'!
description
	^ description! !

!Team methodsFor: 'accessing' stamp: 'mlm 3/9/2001 10:12'!
description: aDescriptionString
	aDescriptionString ifNil: [BusinessRuleException signal: 'Tried to add nil description for team.'].
	description _ aDescriptionString withBlanksTrimmed.
	self changed: #description! !

!Team methodsFor: 'accessing' stamp: 'mlm 3/18/2001 13:48'!
isFormatMultipleChair
	^ self doGetFormat = self formatMultipleChair! !

!Team methodsFor: 'accessing' stamp: 'mlm 3/18/2001 13:48'!
isFormatNoChair
	^ self doGetFormat = self formatNoChair! !

!Team methodsFor: 'accessing' stamp: 'mlm 3/18/2001 13:49'!
isFormatSingleChair
	^ self doGetFormat = self formatSingleChair! !

!Team methodsFor: 'accessing' stamp: 'mlm 3/18/2001 13:48'!
setFormatMultipleChair
	self doSetFormat: self formatMultipleChair! !

!Team methodsFor: 'accessing' stamp: 'mlm 3/18/2001 13:48'!
setFormatNoChair
	self doSetFormat: self formatNoChair! !

!Team methodsFor: 'accessing' stamp: 'mlm 3/18/2001 13:48'!
setFormatSingleChair
	self doSetFormat: self formatSingleChair! !


!Team methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:12'!
addTeamMember: aTeamMember
	aTeamMember ifNil: [BusinessRuleException signal: 'Tried to add nil team member.'].
	aTeamMember addTeam: self! !

!Team methodsFor: 'collaboration-accessing' stamp: 'mlm 9/25/2000 18:07'!
chairs
	^ self teamMembers select: [:aTeamMember | aTeamMember isRoleChair].! !

!Team methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:13'!
removeTeamMember: aTeamMember
	aTeamMember ifNil: [BusinessRuleException signal: 'Tried to remove nil team member.'].
	aTeamMember removeTeam: self! !

!Team methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:13'!
teamMemberFor: aPerson
	^ self teamMembers detect: [:aTeamMember | aTeamMember person = aPerson] ifNone: []! !

!Team methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:34'!
teamMembers
	^ teamMembers! !


!Team methodsFor: 'collaboration-rules' stamp: 'mlm 3/17/2001 16:39'!
testAddTeamMember: aTeamMember
	aTeamMember isRoleChair ifTrue: [self testCanBeChair: aTeamMember]! !

!Team methodsFor: 'collaboration-rules' stamp: 'mlm 3/9/2001 10:21'!
testCanBeChair: aTeamMember
	self isFormatMultipleChair ifTrue: [^ self].
	self isFormatNoChair ifTrue: [BusinessRuleException signal:
		'Tried to add chair team member to no chairs team.'].
	self chairs size = 1 ifTrue: [BusinessRuleException signal: 
		'Tried to add another chair team member to a single chair team.']! !


!Team methodsFor: 'comparing' stamp: 'mlm 3/18/2001 13:49'!
= anObject
	self species = anObject species ifFalse: [^ false].
	self description = anObject description ifFalse: [^ false].
	self doGetFormat = anObject doGetFormat ifFalse: [^ false].
	^ true! !


!Team methodsFor: 'constants' stamp: 'mlm 9/25/2000 17:05'!
formatMultipleChair
	^ Association key: #multiple value: 'multiple chairs'! !

!Team methodsFor: 'constants' stamp: 'mlm 9/25/2000 17:05'!
formatNoChair
	^ Association key: #none value: 'no chairs'! !

!Team methodsFor: 'constants' stamp: 'mlm 9/25/2000 17:05'!
formatSingleChair
	^ Association key: #single value: 'single chair'! !


!Team methodsFor: 'printing' stamp: 'mlm 3/18/2001 13:49'!
printOn: aStream
	aStream nextPutAll: 'Team:'.
	aStream cr; nextPutAll: self description.
	aStream cr; nextPutAll: self doGetFormat value! !


!Team methodsFor: 'private' stamp: 'mlm 9/7/2000 13:25'!
doAddTeamMember: aTeamMember
	self teamMembers add: aTeamMember.
	self changed: #teamMembers! !

!Team methodsFor: 'private' stamp: 'mlm 3/18/2001 13:47'!
doGetFormat
	^ format! !

!Team methodsFor: 'private' stamp: 'mlm 9/7/2000 13:25'!
doRemoveTeamMember: aTeamMember
	self teamMembers remove: aTeamMember ifAbsent: [].
	self changed: #teamMembers! !

!Team methodsFor: 'private' stamp: 'mlm 3/18/2001 13:47'!
doSetFormat: aFormat
	format _ aFormat.
	self changed: #format! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Team class
	instanceVariableNames: ''!

!Team class methodsFor: 'instance creation' stamp: 'mlm 9/23/2000 12:48'!
new
	^ super new initialize! !


!Team class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:15'!
testNoChairTeam
	"Team testNoChairTeam"
	| aTeam |
	aTeam _ self new.
	aTeam setFormatNoChair.
	aTeam description: 'Summer Picnic Planning Team'.
	^ aTeam! !

!Team class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:15'!
testSingleChairTeam
	"Team testSingleChairTeam"
	| aTeam |
	aTeam _ self new.
	aTeam setFormatSingleChair.
	aTeam description: 'Executive Strategy Team'.
	^ aTeam! !

!Team class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:16'!
testTeam
	"Team testTeam"
	| aTeam |
	aTeam _ self new.
	aTeam description: 'System Integration Team'.
	^ aTeam! !


Model subclass: #TeamMember
	instanceVariableNames: 'person team role privileges securityLevel nominations '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Collaboration-Examples'!

!TeamMember methodsFor: 'initialize-release' stamp: 'mlm 3/17/2001 19:06'!
initialize
	person _ nil.
	team _ nil.
	self makeMember.
	securityLevel _ SecurityLevel new.
	nominations _ OrderedCollection new! !


!TeamMember methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:40'!
email
	^ self person email! !

!TeamMember methodsFor: 'accessing' stamp: 'mlm 3/17/2001 10:35'!
isRoleAdmin
	^ self doGetRole = self roleAdmin! !

!TeamMember methodsFor: 'accessing' stamp: 'mlm 3/17/2001 10:35'!
isRoleChair
	^ self doGetRole = self roleChair! !

!TeamMember methodsFor: 'accessing' stamp: 'mlm 3/17/2001 10:35'!
isRoleMember
	^ self doGetRole = self roleMember! !

!TeamMember methodsFor: 'accessing' stamp: 'mlm 3/9/2001 10:15'!
maxNominationsAllowed
	self isRoleChair ifTrue: [^ self maxChairDocuments] ifFalse: [^ self maxDocuments]! !

!TeamMember methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:40'!
name
	^ self person name! !

!TeamMember methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:40'!
securityLevel
	^ securityLevel! !

!TeamMember methodsFor: 'accessing' stamp: 'mlm 9/23/2000 11:40'!
title
	^ self person title! !


!TeamMember methodsFor: 'accessing-rules' stamp: 'mlm 3/9/2001 10:15'!
testSetRoleChair
	self isRoleChair ifTrue: [^ self].
	self team ifNotNil: [self team testCanBeChair: self]! !


!TeamMember methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:14'!
addPerson: aPerson
	aPerson ifNil: [BusinessRuleException signal: 'Tried to add nil person.'].
	self testAddPerson: aPerson.
	self doAddPerson: aPerson.
	aPerson doAddTeamMember: self! !

!TeamMember methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:15'!
addTeam: aTeam
	aTeam ifNil: [BusinessRuleException signal: 'Tried to add nil team.'].
	self testAddTeam: aTeam.
	aTeam testAddTeamMember: self.
	self doAddTeam: aTeam.
	aTeam doAddTeamMember: self! !

!TeamMember methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:40'!
nominations
	^ nominations! !

!TeamMember methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:40'!
person
	^ person! !

!TeamMember methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:16'!
removePerson: aPerson
	aPerson ifNil: [BusinessRuleException signal: 'Tried to remove nil person.'].
	self testRemovePerson: aPerson.
	self doRemovePerson: aPerson.
	aPerson doRemoveTeamMember: self! !

!TeamMember methodsFor: 'collaboration-accessing' stamp: 'mlm 3/9/2001 10:16'!
removeTeam: aTeam
	aTeam ifNil: [BusinessRuleException signal: 'Tried to remove nil team.'].
	self testRemoveTeam: aTeam.
	aTeam testRemoveTeamMember: self.
	self doRemoveTeam: aTeam.
	aTeam doRemoveTeamMember: self! !

!TeamMember methodsFor: 'collaboration-accessing' stamp: 'mlm 9/23/2000 11:40'!
team
	^ team! !


!TeamMember methodsFor: 'collaboration-rules' stamp: 'mlm 3/17/2001 16:37'!
testAddConflictBetween: aPerson and: aTeam
	(aTeam teamMemberFor: aPerson) ifNotNil: [BusinessRuleException signal:
		'Tried to add person twice to team.']! !

!TeamMember methodsFor: 'collaboration-rules' stamp: 'mlm 4/11/2001 10:45'!
testAddNomination: aNomination
	self hasNominatePrivilege ifFalse: [BusinessRuleException signal:
		'Team member cannot nominate.'].
	self countNominationsPerPeriod >= self maxNominationsAllowed 
		ifTrue: [BusinessRuleException signal:
			'Team member cannot nominate. Too many nominations.']! !

!TeamMember methodsFor: 'collaboration-rules' stamp: 'mlm 3/17/2001 16:29'!
testAddPerson: aPerson
	self person ifNotNil: [BusinessRuleException signal: 'Team member already has a person.'].
	aPerson hasValidEmail ifFalse: [BusinessRuleException signal:
		'Tried to add person with invalid email.'].
	self team ifNotNil: [self testAddConflictBetween: aPerson and: self team]! !

!TeamMember methodsFor: 'collaboration-rules' stamp: 'mlm 3/17/2001 16:53'!
testAddTeam: aTeam
	self person ifNotNil: [self testAddConflictBetween: self person and: aTeam]! !

!TeamMember methodsFor: 'collaboration-rules' stamp: 'mlm 3/17/2001 16:39'!
testRemovePerson: aPerson
	self person = aPerson ifFalse: [BusinessRuleException signal:
		'Tried to remove different person.'].
	self nominations isEmpty ifFalse: [BusinessRuleException signal: 
		'Cannot remove person while nominations exist for team member.'].
	self team ifNotNil: [BusinessRuleException signal:
		'Team member on team cannot remove person.']! !

!TeamMember methodsFor: 'collaboration-rules' stamp: 'mlm 3/9/2001 10:33'!
testRemoveTeam: aTeam
	self team = aTeam ifFalse: [BusinessRuleException signal: 'Tried to remove different team.']! !


!TeamMember methodsFor: 'comparing' stamp: 'mlm 3/17/2001 16:55'!
= anObject 
	self species = anObject species ifFalse: [^ false].
	self doGetRole = anObject doGetRole ifFalse: [^ false].
	self person = anObject person ifFalse: [^ false].
	self team = anObject team ifFalse: [^ false].
	^ true! !


!TeamMember methodsFor: 'constants' stamp: 'mlm 9/23/2000 11:41'!
maxChairDocuments
	^ 10! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/23/2000 11:41'!
maxDocuments
	^ 5! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/23/2000 11:42'!
nominationsTimePeriod
	^ 30! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/25/2000 18:14'!
privilegeDefaultMask
	^ 0! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/25/2000 18:14'!
privilegeDeleteMask
	^ 1! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/25/2000 18:15'!
privilegeNominateMask
	^ 2! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/25/2000 18:02'!
roleAdmin
	^ Association key: #admin value: 'Admin'! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/25/2000 18:02'!
roleChair
	^ Association key: #chair value: 'Chair'! !

!TeamMember methodsFor: 'constants' stamp: 'mlm 9/25/2000 18:02'!
roleMember
	^ Association key: #member value: 'Member'! !


!TeamMember methodsFor: 'domain services' stamp: 'mlm 4/11/2001 10:47'!
countNominationsPerDays: anInteger
	| endDate |
	self nominations isEmpty ifTrue: [^ 0].
	endDate _ Date today subtractDays: anInteger.
	^ (self nominations select: [:aNomination | aNomination nominationDate > endDate]) size! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 9/23/2000 11:42'!
countNominationsPerPeriod
	^ self countNominationsPerDays: self nominationsTimePeriod! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 3/17/2001 10:31'!
grantDeletePrivilege
	self doSetPrivileges: (self doGetPrivileges bitOr: self privilegeDeleteMask)! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 3/17/2001 10:32'!
grantNominatePrivilege
	self doSetPrivileges: (self doGetPrivileges bitOr: self privilegeNominateMask)! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 3/18/2001 14:34'!
makeAdmin
	self doSetRoleAdmin.
	self grantNominatePrivilege.
	self revokeDeletePrivilege! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 3/18/2001 14:35'!
makeChair
	self doSetRoleChair.
	self grantNominatePrivilege.
	self grantDeletePrivilege! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 3/18/2001 14:35'!
makeMember
	self doSetRoleMember.
	self doSetPrivileges: self privilegeDefaultMask! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 3/17/2001 10:32'!
revokeDeletePrivilege
	self hasDeletePrivilege
		ifTrue: [self doSetPrivileges: (self doGetPrivileges bitXor: self privilegeDeleteMask)]! !

!TeamMember methodsFor: 'domain services' stamp: 'mlm 3/17/2001 10:32'!
revokeNominatePrivilege
	self hasNominatePrivilege
		ifTrue: [self doSetPrivileges: (self doGetPrivileges bitXor: self privilegeNominateMask)]! !


!TeamMember methodsFor: 'printing' stamp: 'mlm 3/17/2001 10:35'!
printOn: aStream
	aStream nextPutAll: 'TeamMember:'.
	aStream cr; nextPutAll: self doGetRole value.
	self securityLevel printOn: aStream cr.
	self person printOn: aStream cr.
	self team printOn: aStream cr! !


!TeamMember methodsFor: 'testing' stamp: 'mlm 3/17/2001 10:33'!
hasDeletePrivilege
	^ self doGetPrivileges anyMask: self privilegeDeleteMask! !

!TeamMember methodsFor: 'testing' stamp: 'mlm 3/17/2001 10:33'!
hasNominatePrivilege
	^ self doGetPrivileges anyMask: self privilegeNominateMask! !

!TeamMember methodsFor: 'testing' stamp: 'mlm 9/27/2000 13:45'!
hasNomination: aNomination
	^ self nominations includes: aNomination! !

!TeamMember methodsFor: 'testing' stamp: 'mlm 9/26/2000 12:55'!
hasValidEmail
	^ self person hasValidEmail! !


!TeamMember methodsFor: 'private' stamp: 'mlm 12/3/1999 16:34'!
doAddNomination: aNomination
	self nominations add: aNomination.
	self changed: #nominations! !

!TeamMember methodsFor: 'private' stamp: 'mlm 8/23/2000 10:15'!
doAddPerson: aPerson
	person _ aPerson.
	self changed: #person! !

!TeamMember methodsFor: 'private' stamp: 'mlm 9/11/2000 13:40'!
doAddTeam: aTeam
	team _ aTeam.
	self changed: #team! !

!TeamMember methodsFor: 'private' stamp: 'mlm 3/17/2001 10:31'!
doGetPrivileges
	^ privileges! !

!TeamMember methodsFor: 'private' stamp: 'mlm 3/17/2001 10:31'!
doGetRole
	^ role! !

!TeamMember methodsFor: 'private' stamp: 'mlm 12/15/1999 14:56'!
doRemoveNomination: aNomination
	self nominations remove: aNomination ifAbsent: [].
	self changed: #nominations! !

!TeamMember methodsFor: 'private' stamp: 'mlm 2/27/2001 14:36'!
doRemovePerson: aPerson
	person _ nil.
	self changed: #person! !

!TeamMember methodsFor: 'private' stamp: 'mlm 9/18/2000 14:27'!
doRemoveTeam: aTeam
	team _ nil.
	self changed: #team! !

!TeamMember methodsFor: 'private' stamp: 'mlm 3/17/2001 10:30'!
doSetPrivileges: bits
	privileges _ bits.
	self changed: #privileges! !

!TeamMember methodsFor: 'private' stamp: 'mlm 3/18/2001 15:25'!
doSetRole: aRole
	role _ aRole.
	self changed: #role! !

!TeamMember methodsFor: 'private' stamp: 'mlm 3/18/2001 15:26'!
doSetRoleAdmin
	self doSetRole: self roleAdmin! !

!TeamMember methodsFor: 'private' stamp: 'mlm 3/18/2001 15:27'!
doSetRoleChair
	self testSetRoleChair.
	self doSetRole: self roleChair! !

!TeamMember methodsFor: 'private' stamp: 'mlm 3/18/2001 15:27'!
doSetRoleMember
	self doSetRole: self roleMember! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TeamMember class
	instanceVariableNames: ''!

!TeamMember class methodsFor: 'instance creation' stamp: 'mlm 9/15/2000 16:34'!
new
	BusinessRuleException signal: 'Cannot create team member without a person and a team.'! !

!TeamMember class methodsFor: 'instance creation' stamp: 'mlm 3/17/2001 10:18'!
newWith: aPerson and: aTeam
	| aTeamMember |
	aTeamMember _ super new initialize.
	aTeamMember addPerson: aPerson.
	[aTeamMember addTeam: aTeam]
		on: BusinessRuleException
		do: [:ex | aPerson doRemoveTeamMember: aTeamMember.
			ex signal].
	^ aTeamMember! !


!TeamMember class methodsFor: 'examples' stamp: 'mlm 3/17/2001 10:19'!
testAdmin
	"TeamMember testAdmin"
	| aTeamMember |
	aTeamMember _ self newWith: Person testPerson and: Team testTeam.
	aTeamMember makeAdmin.
	^ aTeamMember! !

!TeamMember class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:08'!
testChair
	"TeamMember testChair"
	| aTeamMember |
	aTeamMember _ self newWith: Person testPerson and: Team testTeam.
	aTeamMember makeChair.
	aTeamMember securityLevel setLevelHigh.
	^ aTeamMember! !

!TeamMember class methodsFor: 'examples' stamp: 'mlm 3/17/2001 10:19'!
testNoNominate
	"TeamMember testNoNominate"
	| aTeamMember |
	aTeamMember _ self newWith: Person testPerson and: Team testTeam.
	aTeamMember revokeNominatePrivilege.
	^ aTeamMember! !

!TeamMember class methodsFor: 'examples' stamp: 'mlm 3/17/2001 12:09'!
testSecret
	"TeamMember testSecret"
	| aTeamMember |
	aTeamMember _ self newWith: Person testPerson and: Team testTeam.
	aTeamMember grantNominatePrivilege.
	aTeamMember securityLevel setLevelSecret.
	^ aTeamMember! !
